//
//  SignatureProtocol.h
//  IOS-SDK
//
//  Created by Vital on 2/05/16.
//  Copyright © 2016 Zetes SA/NV. All rights reserved.
//
#import <Foundation/Foundation.h>

#pragma once

/*! @define */
#define E_WRONG_PIN_TYPE               -10
/*! @define */
#define E_PIN_TOO_SHORT                -11
/*! @define */
#define E_PIN_TOO_LONG                 -12
/*! @define */
#define E_PIN_BLOCKED                  -13
/*! @define */
#define E_PIN_CANCELLED                -14
/*! @define */
#define E_PIN_INCORRECT                -20
/*! @define */
#define E_PIN_1_ATTEMPT                -21
/*! @define */
#define E_PIN_2_ATTEMPTS               -22
/*! @define */
#define E_PIN_3_ATTEMPTS               -23
/*! @define */
#define E_SIGNATURE_FAILED             -24
/*! @define */
#define E_ACCESS                       -25
/*! @define */
#define E_DIGEST_ALGO_UNSUPPORTED      -26
/*! @define */
#define E_DIGEST_LEN                   -27
/*! @define */
#define E_UNSUPPORTED                  -28
/*! @define */
#define E_PIN_TIMEOUT                  -29
/*! @define */
#define E_PIN_LENGTH                   -30


/*! @define */
#define DIGEST_ALGO_SHA1               1
/*! @define */
#define DIGEST_ALGO_SHA256             2
/*! @define */
#define DIGEST_ALGO_SHA384             3
/*! @define */
#define DIGEST_ALGO_SHA512             4



/*!
 @discussion A list of signature types.
 @enum example enum
 @constant SignatureTypeAuthentication Signature by authentication key.
 @constant SignatureTypeNonRepudiation Signature by non-repudiation key.
 */
typedef NS_ENUM(NSInteger, SignatureType) {
   SignatureTypeAuthentication = 1,
   SignatureTypeNonRepudiation = 2
};


@protocol SignatureProtocol <NSObject>

/*!
 @discussion
 The private keys are protected by a PIN. After a successful non-repudiation pin validation, the non-repudiation key usage counter is set to 1 which means this key can be used only once. The authentication pin has no limit on the usage counter for the authentication key.
 
 @param pinData Pin to logon for use of a specific key
 @param signatureType Type of key to load for prepare specific type of signature
 @result
 0 - OK
 negative value for pin errors, see list of E_PIN_... errors
 
 */
- (int) verifyPin:(NSData*) pinData forSignatureType:(SignatureType) signatureType;

/*!
 @discussion
 This method returns a PKSCS1 type raw signature. The pkcs1 headers are added by this function before the signature is calculated. Result length depends on the keylength, 128 bytes (1024 bit) or 256 bytes (2048 bit)
 
 @param hash input hash buffer to sign
 @param digestAlgo algorithm specifying the type of hash
 @param signature output block signed by card
 0 - OK
 negative value for pin errors, see list of errors constants
 */
- (int) pkcs1Sign:(NSData*) hash digestAlgo:(NSInteger) digestAlgo signature:(NSMutableData*) signature;

/*!
 @discussion
 This method returns a pkcs1 RSA or EC  raw signature. The pkcs1 headers are added by this function before the signature is calculated. Result length depends on the keylength
 
 @param hash input hash buffer to sign
 @param digestAlgo algorithm specifying the type of hash
 @param signature output block signed by card
 0 - OK
 negative value for pin errors, see list of errors constants
 */
- (int) sign:(NSData*) hash digestAlgo:(NSInteger) digestAlgo signature:(NSMutableData*) signature;


/*!
 @discussion
 This method returns a CAdES-BES type signature.
 
 @param data input hash/detached to sign (if detached=YES, data should be digest from plain text)
 @param digestAlgo algorithm specifying the type of hash
 @param detached flag to indicate the plain data is not part if the CMS signature (if YES, data should only contain hash value from plain input)
 @param cms CMS (=Cryptographic Message Syntax) structure in CAdES-BES DER format signed by eid card inluding certificate chain up to root
 0 - OK
 negative value for pin errors, see list of errors constants
 */
- (int) cmsSign:(NSData*) data digestAlgo:(NSInteger) digestAlgo isDetached:(BOOL) detached cms:(NSMutableData*) cms;


@end
