//
//  Reader.h
//  IOS-SDK
//
//  Created by Vital on 28/04/16.
//  Copyright © 2016 Zetes SA/NV. All rights reserved.
//

/*!
 @header Reader
 @discussion This class is an easy to use interface for eid cards in Belgium.
 @copyright (c) 2013-2019 Zetes SA/NV. All rights reserved.
 @updated 2018-12-08
 @meta http-equiv=”refresh” content=”0;http://www.apple.com”
 */

#import <Foundation/Foundation.h>
#import "ReaderProtocol.h"

@protocol ReaderDelegate <NSObject>
@optional

- (void) readerDidChange:(BOOL) attached;
- (void) cardDidChange:(BOOL) attached;
- (void) didDetectReader:(NSString*) reader;

@end


/*! @define */
#define E_READER_NOT_PRESENT           -1
/*! @define */
#define E_CARD_NOT_PRESENT             -2
/*! @define */
#define E_CARD_NOT_SUPPORTED           -3
/*! @define */
#define E_CARD_READ_ERR                -4
/*! @define */
#define E_READER_NOT_SUPPORTED         -5
/*! @define */
#define E_FUNCTION_NOT_SUPPORTED       -6



/*!
 @discussion
 Base class for reading public eID data.
 See also ReaderProtocol.h for eID reading functions.
 
 To integrate this SDK add following to your project:
 
 - other linker flags: -ObjC
 - ExternalAccessory.framework
 - CoreBluetooth.framework
 - libc++.tbd
 - Supported External Accessory Protocols: "com.zetes.sipirom": This should be added to Info.pList->Supported External Accessory Protocol (in case you want support for blue bluetooth (v3) reader
 - Required Background modes: "App communicates with an accessory")
 
 The External Accessory framework is referenced by the static lib, but your application does not need
 to make any calls directly to this framework.
 
 Example usage in Objectiv C:
 
 Reader reader = [[Reader alloc] init];
 
 int iReturn = [reader open];
 
 NSString firstName = reader.firstName;
 NSString lastName = reader.lastName;
 
 [reader close];
 
 */
@interface Reader : NSObject  <ReaderProtocol>

- (id) initWithPreferredReader: (NSString*) name OEM:(NSString*) oemVersion;
- (id) initWithPreferredReader: (NSString*) name OEM:(NSString*) oemVersion type:(NSString*) type;
- (id) initWithPreferredReader: (NSString*) name OEM:(NSString*) oemVersion type:(NSString*) type minRSSI:(int) minRequiredSignal;

- (void) startScan;
- (void) stopScan;

/*!
 @method open
 @discussion
 Establishes a connection with the reader and an eID card.
 This method also expects a card inserted already, or an appropriate error will be returned.
 
 @result
 0                        on success
 E_READER_NOT_PRESENT     no reader attached
 E_READER_NOT_SUPPORTED   attached reader is not supported
 E_CARD_NOT_PRESENT       no card present in reader
 E_CARD_NOT_SUPPORTED     card not supported by reader
 E_CARD_READ_ERR          error reading card data
 
 */
- (int) open;

/*!
 @method close
 @discussion
 Closes a connection with the reader and an eID card.
 */
- (void) close;

/*!
 @method beginTransaction
 @discussion
 starts a transaction with a reader.
 */
- (int) beginTransaction;

/*!
 @method endTransaction
 @discussion
 ends a transaction with a reader.
 */
- (int) endTransaction;


/*!
 @method supportsPinPad
 @discussion
 returns TRUE if reader has a pinpad
 @result
 TRUE - FALSE
*/
- (BOOL) supportsPinPad;

/*!
 @method version
 @discussion
 Returns the version
 
 @result
 String containing version info
 */

- (NSString*) version;

@property (nonatomic, weak) id <ReaderDelegate> delegate;

@end
